	subroutine sb_initsvp(mtot,nchim,nom_elem,nucleo,version)

!  Version 2.5 (08/08/2025)
!  Reading of options, data and preparation of
!  tables needed for the computation of radiative accelerations (sb_grad2).
!  A complete discussion about atomic diffusion (and radiative accelerations, among
!  many other subjects) can be found in the book:
!  "Atomic diffusion in Stars",2015, G. Michaud and G. Alecian and J. Richer,
!  Astronomy and Astrophysics Library, Springer International Publishing, Switzerland.
!
! Details about the SVP method, and presentation of physics used in these programs are 
! in Alecian & LeBlanc, 2020 MNRAS.

!  Program by:
!     Georges ALECIAN
!     LUTH, Observatoire de Paris, CNRS, PSL
!     F-92190 MEUDON, FRANCE
!     email: georges.alecian@obspm.fr

	Implicit None

	Real(DP), Intent(In) :: mtot
      Integer, Intent(In) :: nchim
      Character(Len=4), Dimension(nchim), Intent(In) :: nom_elem
      Real(DP), Dimension(nchim), Intent(In) :: nucleo
	character(64)           :: version,version_code

	integer                                  :: i,j,k
	
! local variables
	integer, parameter                      :: nsvp=300,nlist=100
	integer                                 :: iunit_tdb,iunit_svp,izn,ios
	integer                                 :: icompte,ic,icf,numion,icsvp
	integer                                 :: eflag,iflag
	integer, dimension(nsvp)                :: np,iz,numa
	integer, dimension(3,nsvp)              :: n_cross
	integer, dimension(nlist)               :: mass_svp,numions
	integer, dimension(nsvp,nlist)          :: nps,izs
	character(1)                            :: b
	character(4)                            :: neutre,neutreold
	character(2), dimension(nsvp)           :: el
	character(4), dimension(nsvp,nlist)     :: listis
	character(4), dimension(nsvp)           :: listi
	character(64), dimension(nlist)         :: listf
	real, dimension(nsvp,nlist)             :: phis,psis,xis,alphs,acs,bets
	real, dimension(nsvp)                   :: phi,psi,xi,alph,ac,bet,absol
	real(KIND=KIND(0D0))                    :: mass_st_sol
	real(KIND=KIND(0D0)), dimension(nlist)  :: X,F1,F2,F3,F4,F5,F6

!
! igrad  (if =0 g_rad of unidentified elements will be set equal to zero)
!        (if =1 g_rad of unidentified elements will be set equal to -local_gravity)
! tgrad  (if =0 g_rad of identified elements will be set equal to zero)
!        (if =1 g_rad of identified elements will be set equal to -local_gravity)
!        (if =2 g_rad of identified elements will be computed)
! np: nb of protons in nucleus
! iz: ionization degree (0 for neutral)
! listi: name of the ion (ex: C1 = neutral carbone)
! numion: nb of ions with available SVP data
!**************************


	OPEN (UNIT = 12,
     &        FILE = 'svp_initialize.job',
     &        STATUS = 'replace')

	version_code='08/08/2025 - 13:10'
	write(12,*) 'svp_codes ==> version of :',version_code
	write(12,*) 'development mode DEV=',DEV ! T or F (change in mod_svp.f)

	
	print*,'svp_codes ==> version of :',version_code
	print*,' development mode DEV=',DEV ! T or F (change in mod_svp.f)

! units for reading
	iunit_tdb=70
	iunit_svp=71

! initializations
	id_ppxi  =  0
	isotops  =  0
	phistar  =  0.
	psistar  =  0.
	xistar   =  0.
	alphstar = -0.5
	acstar   =  1.
	betstar  =  0.

	print*, 'igrad=',igrad
	write(12,*) 'igrad=',igrad
	print*, 'tgrad=',tgrad
	write(12,*) 'tgrad=',tgrad
	print*, '=>'
	write(12,*) '=>'

	if(igrad.eq.0) then
		print*,'g_rad of unidentified elements will be zero'
		write(12,*)'g_rad of unidentified elements will be zero'
	else if(igrad.eq.1) then
		print*,'g_rad of unidentified elements will be -local_gravity'
		write(12,*)'g_rad of unidentified elements will be -local_gravity'
	end if
	if(tgrad.eq.0) then
		print*,'g_rad of identified elements will be zero'
		write(12,*) 'g_rad of identified elements will be zero'
	else if(tgrad.eq.1) then
		print*,'g_rad of identified elements will be -local_gravity'
		write(12,*) 'g_rad of identified elements will be -local_gravity'
	else if(tgrad.eq.2) then
		print*,'g_rad of identified elements will be computed'
		write(12,*) 'g_rad of identified elements will be computed'
	end if
	print*,'(identified = in SVP tables)'
	print*
	write(12,*) '(identified = in SVP tables)'
	write(12,*)

!***	

! Example of the isotops() matrix that is going to be built:
!         H1  He3  He4  C12  C13  N14  N15  O16  O17  Fe56
!  H1     1    0    0    0    0    0    0    0    0    0
! He3     0    1    1    0    0    0    0    0    0    0
! He4     0    1    1    0    0    0    0    0    0    0
! C12     0    0    0    1    1    0    0    0    0    0
! C13     0    0    0    1    1    0    0    0    0    0
! N14     0    0    0    0    0    1    1    0    0    0
! N15     0    0    0    0    0    1    1    0    0    0
! O16     0    0    0    0    0    0    0    1    1    0
! O17     0    0    0    0    0    0    0    1    1    0
!Fe56     0    0    0    0    0    0    0    0    0    1

	j=1
	do while (j.le.nchim)
		icompte=0
		do k=1,nchim
			if(k.lt.j) then
				isotops(j,k)=0
			else if(k.eq.j) then
				isotops(j,k)=1
			else
				if(zi(j).eq.zi(k)) then
					isotops(j,k)=1
					icompte=icompte+1
				else
					isotops(j,k)=0
				end if
			end if
		end do
		ic=1
		do while (ic.le.icompte)
			j=j+1
			do k=1,nchim
				isotops(j,k)=isotops(j-1,k)
			end do
			ic=ic+1
		end do
		j=j+1
	end do
	
	Write(12,'(/,"isotops matrix",/,10x,50(1x,a4))')(nom_elem(j),j=1,nchim)
	do k=1,nchim
		Write(12,'(a4,6x,50(2x,i1,2x))') ADJUSTR(nom_elem(k)),
     +					(isotops(j,k),j=1,nchim)
	end do
	write(12,*)
! Reading of the files list providing the SVP parameters:
	OPEN (UNIT = iunit_svp,
     +      FILE = TRIM(nomch)//TRIM(ADJUSTL(version))//
     +             '/SVP_tables/list_svpf.dat',
     +      STATUS = 'old',
     +      IOStat=ios
     +      )
	If ( ios /= 0 ) STOP " Pb- liste_svp.data"
	icf=0
	read(iunit_svp,*)
	read(iunit_svp,*)
	
	do i=1,100
		read(iunit_svp,'(a)',end=90) listf(i)
		icf=icf+1
	end do
90	continue
	close(iunit_svp)

! Determination of available stellar masses:
	do i=1,icf
		read(listf(i),'(7x,i4)',iostat=ios) mass_svp(i)
		if(ios/=0) then
			print*,'Identification problem for mass_svp => stop'
			stop
		end if
	end do

	mass_st_sol=NINT((mtot/msol)*100.)
	print*,'mass_st_sol=',mass_st_sol  ! new print
	write(12,'(/,"Mass of the star (in unit of solar mass)=",f10.3,/)')
     +	real(NINT((mtot/msol)*100))/100.
	if((NINT(mass_st_sol).lt.mass_svp(1)).or.
     +	(NINT(mass_st_sol).gt.mass_svp(icf))) then
		print*
		print*,'CAUTION: mtot out of limits in sb_initsvp:'
		print*,'mtot*100/msol=',mass_st_sol
		print*,'min*100=',mass_svp(1)
		print*,'max*100=',mass_svp(icf)
		print*
	end if

! Reading of the SVP parameters	
	do i=1,icf
		OPEN (UNIT = iunit_svp,
     +      FILE = TRIM(nomch)//TRIM(ADJUSTL(version))//
     +             '/SVP_tables/'//listf(i),
     +      STATUS = 'old',
     +      IOStat=ios
     +      )
	If ( ios /= 0 ) STOP " Pb- svp_*.data"

		read(iunit_svp,'(a1)') b
		numions(i)=0
		do j=1,1000
		   read(iunit_svp,'(3i4,1x,a4,6e12.3)',end=100) 
     +	        nps(j,i),izs(j,i),izn,listis(j,i),
     +	        phis(j,i),psis(j,i),xis(j,i),
     +	        alphs(j,i),acs(j,i),bets(j,i)

		   numions(i)=numions(i)+1

		end do
100		continue
		close(unit=iunit_svp)
	end do
10	format(i4,a1,i4,a1,e10.3,a1,e10.3,a1,i4,a1,a4)

! count check of the read data
	do i=2,icf
		if(numions(i-1).ne.numions(i)) then
			print*,'i,numions: ',i,numions(i-1),numions(i)
			print*, 'Anomaly for numions => STOP'
			stop
		end if
		do j=1,numions(i)
			if(listis(j,i-1).ne.listis(j,i)) then
				print*,'i,j, : ',i,j,listis(j,i-1),listis(j,i)
				print*, 'Anomaly for listis => STOP'
				stop
			end if
		end do
	end do	

! preparation
	numion=numions(1)
	n_cross= 0                              ! table(3,nsvp)
	np(1:numion)     = nps(1:numion,1)      ! nb of nucleus protons
	iz(1:numion)     = izs(1:numion,1)      ! charge of the ion
	listi(1:numion)  = listis(1:numion,1)
	
! interpolation of the SVP parameters for the stellar mass of the calling program
	do i=1,icf
		X(i)=float(mass_svp(i))
	end do
	do j=1,numion
		F1(1:icf) = phis(j,1:icf)
		F2(1:icf) = psis(j,1:icf)
		F3(1:icf) = xis(j,1:icf)
		F4(1:icf) = alphs(j,1:icf)
		F5(1:icf) = acs(j,1:icf)
		F6(1:icf) = bets(j,1:icf)

		phi(j)  = FTlin(mass_st_sol,100,X,F1)
		psi(j)  = FTlin(mass_st_sol,100,X,F2)
		xi(j)   = FTlin(mass_st_sol,100,X,F3)

!		phi(j)  = FTlinlog(mass_st_sol,100,X,F1) !non-distribuee
!		psi(j)  = FTlinlog(mass_st_sol,100,X,F2) !non-distribuee
!		xi(j)   = FTlinlog(mass_st_sol,100,X,F3) !non-distribuee

		alph(j) = FTlin(mass_st_sol,100,X,F4)
		ac(j)   = FTlin(mass_st_sol,100,X,F5)
		bet(j)  = FTlin(mass_st_sol,100,X,F6)
	end do

! Cross-identification of elements. One sets id_ppxi(j)=1 if at least one ion is OK.
! We assume that atomic transition properties that are relevant for g_rad are 
! the identical for all the isotopes of a given element.
		write(12,*) 
     +	'Final SVP table for this star (same table for all isotops):'
		write(12,'(4x,a,5x,a,8x,a)') 'np  ion   ',
     +	'phistar    psistar    xistar','  alphstar    acstar    betstar'

	icsvp=0
	neutre   =''
	neutreold=''
	do j=1,nchim
		iflag=0
		do k=1,numion
			if(zi(j).eq.np(k)) then
				if(izs(k,1).le.np(k)) iflag=iflag+1
				if(iflag.eq.1) then
					neutreold=neutre
					neutre=listis(k,1)
				endif
				n_cross(1,k)       = iz(k)    !match svp/cesam
				n_cross(2,k)       = j        !match svp/cesam
				n_cross(3,k)       = n_cross(3,k) +1      ! nb.isotopes
				id_ppxi(j)         = 1
				phistar(iz(k),j)   = phi(k)
				psistar(iz(k),j)   = psi(k)
				xistar(iz(k),j)    = xi(k)
				alphstar(iz(k),j)  = alph(k)
				acstar(iz(k),j)    = ac(k)
				betstar(iz(k),j)   = bet(k)
				if(neutre.ne.neutreold) then
					write(12,'(3x,i3,2x,a,5x,3es11.3,5x,3es11.3)')
     +				np(k),listis(k,1),phi(k),psi(k),xi(k),
     +				alph(k),ac(k),bet(k)
				endif
 			end if
		end do
	end do
	do j=1,nchim
		icsvp=icsvp+id_ppxi(j)
	end do

! reading of energy levels of ions

	call sb_niv(listi,numion,n_cross,np,iunit_svp,nchim,version)

! Reading of the reference abundances (khi=1) used for SVP parameters.
! Elements for which SVP parameters are not provided do not appear in this file.
	OPEN (UNIT = iunit_svp,
     +      FILE = TRIM(nomch)//TRIM(ADJUSTL(version))//'/abund_ref.dat',
     +      STATUS = 'old',
     +      IOStat=ios
     +      )
	If ( ios /= 0 ) STOP " Pb- abund_ref.dat"
	ic=0
	do k=1,nsvp
		read(iunit_svp,*,end=101) el(k),numa(k),absol(k)
		ic=ic+1
	end do
101	continue
	close (iunit_svp)
	
! initialization C_ref=1. to avoid underflow for elements with id_ppxi=0. 
! Radiative acceleration will not be computed for these elements, and the "reference"
! abundance attributed to them has no physical meaning and will not be used.

	C_ref=1.

! concentration with respect to H for the khi used in sb_grad_2.

! CAUTION ==> To well estimate the saturation effect of bb transitions, one considers the same
! abundance for all the isotopes, because for radiative accelerations all bb transitions of the
! various isotopes of a given element should overlap in interior layers.  Therefore,
! the momentum transfered from the photon field will not depend on the various concentrations of isotopes,
! but on the total abundance of the element. Radiative accelerations of isotopes will only differ
! because of the difference of their atomic mass.
	
	do j=1,nchim
		eflag=0
		do i=1,ic
			if(zi(j).eq.numa(i)) then
				el_svp(j) = el(i)
				C_ref(j)  = absol(i)
				eflag=1
				exit
			end if
		end do
		if((eflag.eq.0).and.(id_ppxi(j).eq.1)) then
			print*
			print*, 'ic,icsvp= ',ic,icsvp
			print*, 'Any isotope of: ',nom_elem(j)
			print*, 'missing reference abundance (check file abund_ref.dat) !'
			print*, '=> STOP in sb_initsvp()'
			stop
		endif
	end do
	do j=2,nchim
		C_ref(j)=C_ref(j)/C_ref(1)
	end do
	C_ref(1)=1.
	close (12)

	end subroutine sb_initsvp
!=======================================================================
	subroutine sb_niv(listi,numion,n_cross,np,iunit_svp,nchim,version)
!  Version 1.2 (G. Alecian, April 2020)
!  This subroutine reads the atomic energy levels to calculate the ions population and
!  to estimate the momentum given to atoms through photoionization.
!  These atomic data have been previously processed to reduce drastically the number of
!  of energy levels through a gathering process which preserves a good accuracy for the present purpose.


	implicit none

! input
	integer, parameter                      :: nsvp=300
	integer                                 :: numion,iunit_svp,nchim
	integer, dimension(nsvp)                :: np
	integer, dimension(3,nsvp)              :: n_cross
	character(4), dimension(nsvp)           :: listi
	character(64)                           :: version

! local variables
	character(319)                          :: nfe
	integer                                 :: i,j,k,ii,jj,ios
	integer                                 :: dim
	character(4)                            :: nom
	integer, dimension(nsvp)                :: jz=0,ns=0,elflag
	real, dimension(99,nsvp)                :: sae,sag,saq
	real, dimension(nsvp)                   :: x=0.

	elflag=0         ! initialization

! Load of all available data
	do k=1,numion
		nfe=TRIM(nomch)//TRIM(ADJUSTL(version))//
     +	'/fused_levels/union'//TRIM(ADJUSTL(listi(k)))//'.niv'
!     +	'../tbion2/tbion2'//TRIM(ADJUSTL(listi(k)))//'.dat'
			
		OPEN (UNIT = iunit_svp,
     +		FILE = nfe,
     +            IOSTAT=ios,
     +		STATUS = 'old')

		if(ios/=0) then
			print*, 'OPEN problem for ',nfe
			print*, '=> STOP'
			stop
		endif

		elflag(k)=1      ! le fichier de l'ion existe

		read(iunit_svp,1075) nom                       ! ion name
		read(iunit_svp,1081) x(k)                      ! ioni. potential (eV)
		read(iunit_svp,1073) jz(k)                     ! charge
		read(iunit_svp,1073) ns(k)                     ! number of levels
		read(iunit_svp,1045) dim                       ! unused
		read(iunit_svp,1062) (sae(i,k),i=1,ns(k))      ! energy (eV)
		read(iunit_svp,1057) (sag(i,k),i=1,ns(k))      ! stat. weight
		read(iunit_svp,1074) (saq(i,k),i=1,ns(k))      ! Main or effective quantum numb.
		close (iunit_svp)

	end do

! Connection to ions for which SVP accelerations will be computed
	do k=1,numion
		if(n_cross(2,k).ne.0) then
			ii = n_cross(1,k)
			jj = n_cross(2,k)
			do j=jj-n_cross(3,k)+1 , jj     ! n_cross(3,k): nb of isotopes
				niv_flag(ii,j) = elflag(k)
				niv_nb(ii,j)   = ns(k)
				niv_z(ii,j)    = jz(k)
				el_pot(ii,j)   = x(k)
				do i=1,ns(k)
					niv_e(i,ii,j) = sae(i,k)
					niv_q(i,ii,j) = saq(i,k)
					niv_g(i,ii,j) = sag(i,k)
				end do
				call gr_gazrare(np(k),jz(k),rar_flag(ii,j))
			end do

		end if
	end do

1045	format(16i5)
1057	format(12f6.0)
1062	format(8F11.3)
1073	format(20i4)
1074	format(12F6.2)
1075	format(1x,a4)
1081	format(16f9.2)

	end subroutine sb_niv
!=======================================================================
	subroutine sb_g_rad2(nchim,nom_elem,nucleo,mtot,rtot,teff,
     +	                     mass,ray,t,nel,ychim,g_rad,dg_rad)

!  Version 2.4 (G. Alecian, Nov 2024)
!  Computes the radiative accelerations and their derivative with respect to the abundance.
!  The subroutine must called for each layer, each time atomic diffusion is computed in the 
!  main calling program.

	implicit none

	integer, Intent(In)       :: nchim
	real(DP), Intent(In)      :: mtot,rtot,teff,mass,ray,t,nel

	character(4), dimension(nchim), Intent(In)    :: nom_elem
	real(DP), dimension(nchim), Intent(In)        :: nucleo,ychim
	
! OUTPUTS:
      Real(DP), DIMENSION(nchim+1), INTENT(out) :: g_rad
      Real(DP), DIMENSION(nchim+1,nchim+1), INTENT(out) :: dg_rad

! For  g_cont and gr_popion3
	real(KIND=KIND(0D0)), dimension(0:pzi) :: fparti,theta
	real(KIND=KIND(0D0)), dimension(99,0:pzi)  :: ff
	real(KIND=KIND(0D0)), dimension(0:pzi,pnchim) :: popi

! local variables
	integer                  :: iso,j,k
	real,parameter           :: pondrare = 1.5
	real(KIND=KIND(0D0))     :: pondt,x,f
	real(KIND=KIND(0D0))	 :: gr_kj,dgr_kj,g_grav,q,b,Ck_s,CX
	real(KIND=KIND(0D0))	 :: gc_kj,dgc_kj,bco

	real(KIND=KIND(0D0)), dimension(nchim)          :: N_chim,khi
	real(KIND=KIND(0D0)), dimension(0:pzi,nchim)    :: C_ion
	real(KIND=KIND(0D0)), dimension(0:pzi,nchim)    :: pond_ion

!**************************

! initialization
	popi=0.
	g_rad  = 0.
	dg_rad = 0.
	if(ychim(1)*t*ray.lt.1.d-50) return
	khi    = 1.
	gc_kj  = 0.
	dgc_kj = 0.

! magnitude of the gravity vector
	if(ray.gt.1.) then
		g_grav=g*mtot*(1.-mass)/ray**2
	else
		return            !  g_rad, dg_rad are set equal to zero at the centre
	end if

! One assumes that radiation frequency of atomic transitions are the same for all the
! isotopes of a given element. This approximation is justified in stellar interiors.
! This implies that saturation effects are the same for all the isotopes. This saturation
! determined by the concentration C_ion(k,j) (in number of atoms with respect to H). 
! Therefore, in the formulae used to estimate the radiate acceleration, the concentration
! must be the same for all the isotopes.
! Notice that: Sum[ychim(j)*nucleo(j)]=1

	do j=1,nchim
		N_chim(j)=0.
		do iso=1,nchim
			N_chim(j)=N_chim(j)+ychim(iso)*isotops(j,iso)
		end do
	end do
	do j=1,nchim
		khi(j)=(N_chim(j)/N_chim(1)) / C_ref(j)
	end do

! weighting factor to compute total acceleration of element. 
! The weight is set equal to 1 by default, and 1.5 for noble gas configuration.
! See Alecian & LeBlanc (2020).

	do j=1,nchim
		if(id_ppxi(j).eq.1) then
			call gr_parti3(t,nel,fparti,ff,j)
			call gr_popion3(t,nel,fparti,ff,popi,j)
			do k=0,nint(zi(j))
				C_ion(k,j)=popi(k,j)*N_chim(j)/N_chim(1)
			end do
			pondt=0.
			do k=0,nint(zi(j))
				pondt = pondt + popi(k,j)*
     +		       (1. + rar_flag(k,j)*(pondrare-1.) )
			end do
			do k=0,nint(zi(j))
				pond_ion(k,j) = popi(k,j)*
     +		             (1. + rar_flag(k,j)*(pondrare-1.) )/pondt
			end do
		end if
	end do

!>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
! constantes:
!                pi**2 * kbol**3 * echarg**2
!   5.57E-05 = ---------------------------------
!             2.* clight**4 * hpl**2 * me * amu
!
!
!               me * mp* clight
!   9.83E-23 = -----------------
!                    2. * pi
!
!   CX : mass fraction of hydrogen
!>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

	if(tgrad.eq.0) then
		do j=1,nchim
			 if(id_ppxi(j).eq.1) then
			 	g_rad(j) = 0.
			 else
				if(igrad.eq.0) then
				 	g_rad(j) = 0.
				else if(igrad.eq.1) then
				 	g_rad(j) = g_grav *
     +				(1-pond_ion(nint(zi(j)),j))    !correction for bare nucleus.
! The bare nucleus cannot have g_rad = -g, and cannot be photoionized.
				end if
			 end if
		end do
	else if(tgrad.eq.1) then
		do j=1,nchim
			 if(id_ppxi(j).eq.1) then
			 	g_rad(j) = g_grav
			 else
				if(igrad.eq.0) then
				 	g_rad(j) = 0.
				 else if(igrad.eq.1) then
				 	g_rad(j) = g_grav *
     +				(1-pond_ion(nint(zi(j)),j))    !correction for bare nucleus.
! The bare nucleus cannot have g_rad = -g, and cannot be photoionized.
				 end if
			 end if
		end do
	else if(tgrad.eq.2) then
		do j=1,nchim

			if(id_ppxi(j).eq.1) then
				call gr_parti3(t,nel,fparti,ff,j)
				call gr_theta(t,nel,fparti,ff,theta,j)
			end if

			CX  = N_chim(1) * nucleo(1)
			q   = 5.57E-05 * ((teff**4)/t) * ((rtot/ray)**2) / nucleo(j)
			b   = 9.83E-23 * nel * (1./sqrt(t)) / CX

			do k=0,nint(zi(j))
			   if(id_ppxi(j).eq.1) then
				if(niv_z(k,j).gt.0) then
					bco = 7.16e-26*(teff**4)*nel*((rtot/ray)**2)
     +	                   /(t**1.5)/(nucleo(j)*niv_z(k,j)*niv_z(k,j))
					gc_kj  = bco * theta(k)
				else
					gc_kj = 0.
				end if

			      if(psistar(k,j).gt.1.E-30) then

				   Ck_s   = b * psistar(k,j)**2


				   gr_kj  = q * phistar(k,j) 
     +					  * (1. + xistar(k,j)*C_ion(k,j))
     +					  * (1. + C_ion(k,j)/Ck_s)**alphstar(k,j)
     +				    + acstar(k,j) * gc_kj
     +					  * (khi(j)/(khi(j)+1.))**betstar(k,j)
			 	   dgr_kj = 0.
				   if((C_ion(k,j) + Ck_s).gt.1.E-30) then
				   	dgr_kj = popi(k,j)*gr_kj* (
     +					xistar(k,j)/(1. + xistar(k,j)*C_ion(k,j))
     +					+ alphstar(k,j)/(C_ion(k,j) + Ck_s) )
				   endif
				   dgc_kj = betstar(k,j)*gc_kj*
     +					(khi(j)**betstar(k,j))
     +					/((khi(j)+1.)**(betstar(k,j)+1))

!		write(37,*) '**** psistar(k,j)>.1.E-30 :',t,k,j,
!     +			dgr_kj,dgc_kj,C_ion(k,j)+Ck_s ! !non-distribuee
				 else

				   gr_kj  =  acstar(k,j) * gc_kj
     +					  * (khi(j)/(khi(j)+1.))**betstar(k,j)
			 	   dgr_kj = 0.
				   dgc_kj = betstar(k,j)*gc_kj*
     +					(khi(j)**betstar(k,j))
     +					/((khi(j)+1.)**(betstar(k,j)+1))
				 end if
				 g_rad(j)    = g_rad(j) + pond_ion(k,j)*gr_kj
				 dg_rad(j,j) = dg_rad(j,j) + pond_ion(k,j)* (
     +				     dgr_kj/N_chim(1) +
     +				     dgc_kj*N_chim(j) )
			   else      ! for unidentified elements
				if(igrad.eq.0) then
			 	   g_rad(j) = 0.
			 	else if(igrad.eq.1) then
			 	   g_rad(j) = g_grav
			 	end if
			   	dg_rad(j,j) = 0.
			   end if
			end do

! In SVP approximation, radiative acceleration tends to zero for T > 10e+7K,
! it is set to 0 when T=1.4e+7.
			x = 1.4e+07/t - 1.
			if(x.ge.0.4) then
				f = 1.
			else if(x.lt.0.) then
				f = 0.
			else
				f = ((x/0.4)**2)*(-2*(x/0.4)+3.)
			end if
			g_rad(j) = f * g_rad(j)
			dg_rad(j,j) = f * dg_rad(j,j)

		end do
	end if



	end subroutine sb_g_rad2
!=======================================================================
	subroutine gr_gazrare(nelem,jz,rarflag)
! Identification of ions with noble gaz configuration (G.Alecian, April 2020)
	implicit none
	integer, dimension (6)      :: zgrar
	integer                     :: nelem,j,jz,rarflag
	data zgrar/2,10,18,36,54,86/
	do j=1,6
		if(zgrar(j).eq.(nelem-jz)) then
			rarflag=1
			exit
		end if
	end do
	end subroutine gr_gazrare
!=======================================================================
	subroutine gr_theta(t,nel,fparti,ff,theta,j)

! Version 1.2 (G.Alecian, April 2020)
! Determination of the Theta function (Eq.17 of G. Alecian and F. LeBlanc 2002,
! MNRAS  332  891-900) for acceleration due to photoionization. This is for all the ions
! of the "j" element, in a given layer at temperature "t".

	implicit none
	integer                     :: j
	real(DP)	                   :: t,nel
	real(KIND=KIND(0D0)), dimension(0:pzi)        :: fparti
	real(KIND=KIND(0D0)), dimension(99,0:pzi)     :: ff

! output
	real(KIND=KIND(0D0)), dimension(0:pzi) :: theta

! local variables
	integer                             :: k,m,ks
	real, parameter                     :: secu=100.
	real(KIND=KIND(0D0))                :: tke,uk,Qk

!=========== initialization
	theta=0.
!===========
	

	tke=t * 8.6173431838E-05     ! temperature en eV

	do k=1,nint(zi(j)) ! loop on all ions (the first ion does not receive any momentum)

		ks = niv_nb(k,j)
		if(ks.ge.1) then
			do m=1,ks   ! loop on energy levels of ion k
				uk  = (el_pot(k-1,j)-niv_e(m,k-1,j))/tke
				if(uk.lt.1.e-2)  CYCLE
				if(uk.gt.20.)    CYCLE
				
				Qk=(uk**3)*(uk/(1.-exp(-uk)) - exp(uk)*log(1.-exp(-uk)))
				Qk= ff(m,k-1) * Qk

				theta(k) = theta(k) + 
     +	              niv_q(m,k-1,j)*(niv_g(m,k-1,j)/niv_g(1,k-1,j)) * Qk
	
			end do
			theta(k)=niv_g(1,k-1,j)/fparti(k)*theta(k)
		end if
	end do

	end subroutine gr_theta
!=======================================================================
	subroutine gr_parti3(t,nel,fparti,ff,j)

! Determination of partition functions of all the ions of the element "j"
! in a given layer at temperature "t".
! Simple approximation having enough accuracy for the present purpose. To be used only
! with the gathered energy levels read with sb_niv. Truncation of energy levels series
! according to the Debye radius.

!  Version 3.0 (G.Alecian, April 2020)

	implicit none
	integer                                    :: j
	real(KIND=KIND(0D0))	                   :: t,nel

! output
	real(KIND=KIND(0D0)), dimension(0:pzi)     :: fparti
	real(KIND=KIND(0D0)), dimension(99,0:pzi)  :: ff

! local variables
	integer                             :: k,m,ks
	real, parameter                     :: Ze=4.8032044E-10,secu=100.
	real, parameter                     :: ajust=5.
	real(KIND=KIND(0D0))                :: tke,Rd,x_C,x_cesam
	real(KIND=KIND(0D0)), dimension(99) :: f,xx

!=========== initialization
	fparti=0.
!===========
	
	Rd = SQRT(23.8112260522911 * t / nel)
! Debye radius (assuming complete ionization, charge density number = 2*ne)
! 23.8112260522911=kB/(8*pi*(e^2)) (same approx. as Michaud et al 2015, Eq. 4.10)

	tke=t * 8.6173431838E-05     ! temperature (eV)
	


	ff = 1.
	do k=0,nint(zi(j))

		ks = niv_nb(k,j)
		xx= 0.
		if(k.eq.0) then 
			x_C = 0.001*1.6E-12
		else
			x_C = real(k)*(Ze**2)/Rd
		endif
		
		do m=1,ks   ! loop on energy levels of ion k

			if(niv_e(m,k,j)/tke.gt.secu) CYCLE   ! m suivant

			x_cesam = ( ( (el_pot(k,j)-niv_e(m,k,j)) *
     +		          1.6E-12/x_C)/ajust - 1.)

			if(x_cesam.ge.4.) then
				ff(m,k) = 1.
			else if(x_cesam.lt.0.) then
				ff(m,k) = 0.
			else
				ff(m,k) = ((x_cesam/4.)**2)*(-2*(x_cesam/4.)+3.)
			end if

			if(m.eq.1) then
				fparti(k) = fparti(k) + 
     +			            niv_g(m,k,j) * exp(-niv_e(m,k,j)/tke)
			else
				fparti(k) = fparti(k) + ff(m,k) *
     +			            niv_g(m,k,j) * exp(-niv_e(m,k,j)/tke)
			end if



		end do


	end do
	
	end subroutine gr_parti3
!=======================================================================
	subroutine gr_popion3(t,nel,fparti,ff,popi,j)

! Computation of relative population of all the ions of "j" in a given layer

!  Version 3.1 (G.Alecian, April 2020)

	implicit none
	integer                                    :: j
	real(KIND=KIND(0D0))	                   :: t,nel

! input
	real(KIND=KIND(0D0)), dimension(0:pzi) :: fparti
	real(KIND=KIND(0D0)), dimension(99,0:pzi)  :: ff
! output
	real(KIND=KIND(0D0)), dimension(0:pzi,pnchim) :: popi

! local variables
	integer                                :: k,m,na0,na,nb
	real, parameter                        :: secu=100.
	real(KIND=KIND(0D0))                   :: tke,r,d,an,c,lolog
	real(KIND=KIND(0D0)), dimension(0:pzi) :: ax,p

!=========== initialisations
	r   = 1.
	d   = 1.
	na0 = 0
	na  = nint(zi(j))
	an  = 1.  
!===========
	
	tke = t * 8.6173431838E-05     ! temperature en eV
	c = (1.30364e+26 * 1.38e-16 * t)**(-1.5) 
! 1.30364e+26 = (2pi*me/h**2)**3/2

	do k=0,nint(zi(j))
	
		if(niv_nb(k,j).ge.1) then		
			nb=k
		else
			na0=k+1
		endif
	end do

	do k=nb,na0+1,-1  ! loop on ions
		if(el_pot(k-1,j)/tke.gt.secu) then
			ax(k) =1.e+35
			nb = k-1
		else
			ax(k) = ff(1,k-1)*(nel/2.)*
     +		      (fparti(k-1)/fparti(k))*c*exp(el_pot(k-1,j)/tke)
		end if
	end do
	
	p=0.
	do k=nb,na0+1,-1
		do m=k,na0+1,-1
			d = d*ax(m)
			r = d + r
		end do
		p(k)=1./r
		if(p(k).lt.1.e-30) then
			p(k)=0.
			r=1.
			d=1.
		else
			na=k
			exit
		end if
	end do

	do k=na-1,na0,-1
		p(k)=p(k+1)*ax(k+1)
	end do

	do k=nint(zi(j)),0,-1
		popi(k,j)=p(k)*an
	end do

	end subroutine gr_popion3
!=======================================================================
	FUNCTION FTlin(T,N,X,F)
!
!	INTERPOLATION lineaire (originellement FT à 3pts et pas non constant)
!	X est le tableau de la variable discrete 
!	F est le tableau de la fonction F(X) 
!	T est la valeur de X a laquelle on veut interpoler (resultat FTlin=F(T))
!	N est la dimension du tableau
c modifs AH 5/2024, GA 11/11/2024

	IMPLICIT DOUBLE PRECISION(A-H,O-Z)
	INTEGER :: N, I, J
	DIMENSION X(N),F(N)

	DO J=1,N
		I=J
!		IF(T-X(J)) 3,2,1  ! fortran obsolete (2024)
		IF((T-X(J)).lt.0.) then
			goto 3
		else if((T-X(J)).eq.0.) then
			goto 2
		else
			goto 1
		endif
2		FTlin=F(J)
		GOTO 1000
1		CONTINUE
	END DO
3	CONTINUE
c interpolation lineaire.  AH 5/2024
	if(i.gt.1)i=i-1
	if(i.ge.n)i=n-1
	if(t-x(i).lt.0)print*,'FTlin : Attention extrapolation'
	if((x(i+1)-x(i)).lt.1.D-20) then
		print*,'division par 0, ',i,x(i+1),x(i)
	end if
	FTlin=f(i)+(t-x(i))*(f(i+1)-f(i))/(x(i+1)-x(i))

1000	CONTINUE
	END FUNCTION FTlin
!!=======================================================================
!	FUNCTION FTlinlog(T,N,X,F) !non-distribuee
!!
!!	INTERPOLATION lineaire du log (originellement FT à 3pts et pas non constant)
!!	X est le tableau de la variable discrete 
!!	F est le tableau de la fonction F(X) 
!!	T est la valeur de X a laquelle on veut interpoler (resultat FTlinlog=10**F(T))
!!	N est la dimension du tableau
!c modifs AH 5/2024, GA 11/11/2024
!
!	IMPLICIT DOUBLE PRECISION(A-H,O-Z)
!	INTEGER :: N, I, J
!	DIMENSION X(N),F(N)
!
!	DO J=1,N
!		I=J
!!		IF(T-X(J)) 3,2,1  ! fortran obsolete (2024)
!		IF((T-X(J)).lt.0.) then
!			goto 3
!		else if((T-X(J)).eq.0.) then
!			goto 2
!		else
!			goto 1
!		endif
!2		FTlinlog=F(J)
!		GOTO 1000
!1		CONTINUE
!	END DO
!3	CONTINUE
!	
!c interpolation lineaire.  AH 5/2024, GA 11/11/2024
!	if(i.gt.1)i=i-1
!	if(i.ge.n)i=n-1
!	if(t-x(i).lt.0)print*,'FTlinlog : Attention extrapolation'
!	FTlinlog=10.**(lolog(f(i))+(t-x(i))*
!     +	(lolog(f(i+1))-lolog(f(i)))/(x(i+1)-x(i)))
!
!	if(FTlinlog.lt.1.E-29) FTlinlog=0.
!	
!1000	CONTINUE
!	END FUNCTION FTlinlog
!=======================================================================
!	function lolog(A) !non-distribuee
!
!	real(KIND=KIND(0D0))   :: lolog,A
!
!	if (A >0.) then
!		lolog = log10(A)
!	else
!		lolog = -30.
!	end if
!
!	return
!	end
!		
