	Program svp_standalone

!
!	Update: 13 Jan 2026
!
!	This is a standalone demo program for the computation of SVP-radiative accelerations.
!	Description of the SVP method may found from thedownload links: 
!	http://gradsvp.obspm.fr
!	This code has been written by G. Alecian (georges.alecian@obspm.fr).
!
!     This new version (Novembre 2024) allows to chose a test model of 1.5 solar mass 
!	(absent from the list models_list.dat) such that the code is forced
!	to interpolate SVP parameters from the provided tables that are established
!	with CESTAM models, making the demo more realistic. This test model was computed 
!	with a  unique TGEC evolution code for only one mass (no interpolation).
!
! Details and discussion about the SVP method, and presentation of the physics used in 
! this program are in Alecian & LeBlanc, 2020 MNRAS.

!	svp_standalone.f contains the main program and subroutines needed to 
!	compute SVP radiative accelerations in a standalone way and for stars being
!	at a given stage of their evolution (at about the middle of the main-sequence), and
!	for which the corresponding model files are given in data_standalone/. sub-directory 
!	(see the readme.txt file).
!	This code uses also the routines that are in the files mod_donvar.f, mod_svp.f,
!	svp_codes.f. When the computation of radiative accelerations (using the SVP method) 
!	has to be implemented in an evolution code, the files mod_donvar.f, mod_svp.f,
!	svp_codes.f have to be compiled and linked with the evolution code. 
!	The svp_standalone program shows how the svp subroutines have to be called.

!	Old version of the routines that are in the files mod_donvar.f, mod_svp.f,
!	svp_codes.f were implemented in the following evolution codes in collaboration
!	with people named in parenthesis:
!	TGEC (S.Theado, and later by A. Hui-Bon-Hoa), CESAM2K (B. Pichon) not validated, CESTAM (M. Deal)

      USE MOD_SVP, Only: initialize_SVP, g_rad_SVP , grav, teff, mass,
     +                   id_ppxi, C_ref,isotops
      USE MOD_DONNEES, only: nomch => nom_chemin, ctes_94, nom_elem, 
     +                       nchim, nucleo, zi
      USE MOD_VARIABLES, ONLY : mtot => mstar , rtot => rstar

      Implicit None

***  variables that are read from .osc model files (output of CESTAM)
***  variables that are read from .xtl model files (output of TGEC)

	integer  :: pzi,pnchim,lm
	parameter  (pzi=26,pnchim=50,lm=5000)

	real(KIND=KIND(0D0)) :: Mnucleo(pnchim),Mzi(pnchim),Mmtot,
     +	Mrtot,Mltot,Mteff,Mmass(lm),Mray(lm),Mt(lm),Mro(lm),
     +	Mkap(lm),Mgrad(lm),Mlum(lm),Mnel(lm),Mxchim(pnchim,lm),Mnp(lm)
	
	integer  :: itot,i,j,k, ios

	real(KIND=KIND(0D0)) :: ltot,ray,t,ro,kap,grad,lum,nel,
     + 		ychim(pnchim)
***

	character(1)   :: skip
	Character(4), Dimension(pnchim) :: nom_elem_bis

*======================================================
*** Input of subroutine initialize_SVP(...) in mod_svp.f
	character(64)           :: version_svp
*** Outputs of subroutine g_rad_SVP(...) in mod_svp.f
	Double Precision, Dimension(:),   Allocatable :: g_rad
	Double Precision, Dimension(:,:), Allocatable :: dg_rad
*====================================================== 
*** local variables
	integer         :: iunit,icouches,iso,nmod
	parameter(nmod=30)
	integer, dimension(nmod)   :: masses_list
	real     :: logT, logPop(0:pzi),temp(lm),Chi(pnchim,lm),khi(pnchim)
	real(KIND=KIND(0D0)) :: lolog,dpz
	integer	         :: massest,trouv
	character(64)        :: NFIa,NFIb,NFIm,version,teststring
      Character(80)        :: ligne
      Character(20),dimension(nmod) :: models_liste
      integer  :: check_inter
*****************************************
2000	format(1x,1p8d10.3)
2001	format(1x,1p15d10.3)
2002	format(1x,i5,'	',f8.4,1p,100('	',d10.3))
2003	format(1x,'i	ychim	logT',30('	',a5,i3))
2004	format(1x,1p8d15.8)

	version='svp_standalone version 4.2, 13/01/2026, 15:28'
	print*, version
	check_inter=0

	OPEN (UNIT = 10,
     &        FILE = 'svp_standalone.job',
     &        STATUS = 'replace')

	write(10,*) version
	write(10,*) 

	print*
	print*,'Check with a model, such that SVP param will be interpolated ?'
	print*
	print*, '*** If 1, the model will be chosen among CESTAM_models for which'
	print*, '    SVP tables have been computed. No interpolation is done.'
	print*, '*** If 2, it is a TGEC model for a 1.5 solar mass star.'
	print*, '    This is a more realistic demo that uses a non-cestam model,'
	print*, '    and forces the code to interpolate SVP parameters to adapt.'
	print*, '*** If 3, it is a TGEC model for a 2 solar masses star.'
	print*, '    This model includes calculation of g_rad of Sc and Ni,'
	print*, '    to check introduction of Sc and Ni in datai_SVP_v2 (2026).'
	print*, '    No interpolation is done.'
	print*
	print*, 'Choose 1, 2 or 3:'
	read(5,*) check_inter
!	print*,check_inter

!****
	If(check_inter.eq.2) then
		print*
		print*,'You have chosen CHECK_INTER=2'

		NFIa='data_standalone/TEST_model/tgec_1p5.xtg'
		open(unit=1,form='formatted',status='old',file=NFia)

		NFIb='data_standalone/TEST_model/tgec_1p5.xtl'
		open(unit=2,form='formatted',status='old',file=NFib)

!		reads TGEC models (main-sequence star)
		call littgec2(
     &	nchim,nom_elem_bis,
     &	Mmtot,Mrtot,Mltot,Mteff,
     &	Mmass,Mray,Mt,Mro,Mkap,Mgrad,Mlum,Mnel,
     &	Mxchim,Mnp,itot,44)

		close(unit=1)
		close(unit=2)

!****
	else if(check_inter.eq.1) then
		print*
		print*,'You have chosen CHECK_INTER=1'
		print*, 'Choose a mass from the list (ex: 100=1Msol, 120=1.2Msol):'
		NFIm='data_standalone/CESTAM_models/models_list.dat'
		open(unit=3,form='formatted',status='old',file=NFim)
		do i=1,nmod
			read(3,*,end=100) masses_list(i),models_liste(i)
			models_liste(i)=TRIM(ADJUSTL(models_liste(i)))
			print*,masses_list(i),'   ',models_liste(i)
		end do
100		close(3)
		read(5,*) massest
		trouv=0
		do i=1,nmod
			if(massest.eq.masses_list(i)) then
				NFIb='data_standalone/CESTAM_models/'//TRIM(ADJUSTL(models_liste(i)))
				trouv=1
				exit
			endif
		end do
		if(trouv.eq.0) then
			print*,'Unidentified mass -> STOP'
			stop
		endif
		print*,'Chosen mass: ',massest,'	file: ',NFIb
	
		open(unit=1,form='formatted',status='old',file=NFib)
!		reads CESAM/CESTAM models (main-sequence star)
		call litcesam(
     &	nchim,nom_elem_bis,
     &	Mmtot,Mrtot,Mltot,Mteff,
     &	Mmass,Mray,Mt,Mro,Mkap,Mgrad,Mlum,Mnel,
     &	Mxchim,Mnp,itot)
!
		close(unit=1)

!****
	else if(check_inter.eq.3) then
		print*
		print*,'You have chosen CHECK_INTER=3'

		NFIa='data_standalone/TEST_model/tgec_2p0.xtg'
		open(unit=1,form='formatted',status='old',file=NFia)

		NFIb='data_standalone/TEST_model/tgec_2p0.xtl'
		open(unit=2,form='formatted',status='old',file=NFib)

!		reads TGEC models (main-sequence star)
		call littgec2(
     &	nchim,nom_elem_bis,
     &	Mmtot,Mrtot,Mltot,Mteff,
     &	Mmass,Mray,Mt,Mro,Mkap,Mgrad,Mlum,Mnel,
     &	Mxchim,Mnp,itot,46)

		close(unit=1)
		close(unit=2)
	end if

! OUTPUTS
! Elements for which SVP radiative acceleration have to be computed:
!   nchim: number of elements (isotopes of a given element are proceeded as 
!          different elements, but have the same radiative acceleration)
!   nom_elem_bis: name of elements (ex: C12 for carbon 12, Al27 for aluminum 27)
! Stellar parameters:
!   Mmtot: mass of the star (g)
!   Mrtot: radius  of the star (cm)
!   Mltot: luminosity  of the star (erg)
!   Mteff: effective temperature
!   itot: total number of layers of the star model
! For each layer:
!   Mmass: mass above the layer (0 at the surface, 1 at the senter)
!   Mray: local radius
!   Mt:  temperature
!   Mro:  density
!   Mkap:  Rosseland opacity
!   Mgrad:  radiative temperature gradient
!   Mlum:  luminosity
!   Mnel: electrons density  (number/cm3)
!   Mxchim: local abundances, mass fraction for each of the nchim elements
!   Mnp: protons density (number/cm3)
!****************************************

	print*,'Model read done !'
	write(10,'(a,es10.3,a)') 'Star mass: ',Mmtot/1.9891d33,' Msol'
	write(10,*) 'Model file: ', NFIb

! output check
	OPEN (UNIT = 20,	
     &        FILE = 'ctrlg_rad.dat',
     &        STATUS = 'replace')

!	OPEN (UNIT = 30,
!     &        FILE = 'popul.dat',  ! ions population
!     &        STATUS = 'replace')

      Allocate( nom_elem(nchim) , nucleo(nchim) , zi(nchim) , Stat=ios )
      If ( ios /= 0 ) STOP " 1 "
	Allocate( g_rad(nchim+1) , dg_rad(nchim+1,nchim+1) , Stat=ios )
      If ( ios /= 0 ) STOP " 2 "

      nom_elem(1:nchim) = nom_elem_bis(1:nchim)

	write(10,*) 'Elements list:'
	write(10,91) nchim,(nom_elem(i),i=1,nchim)

	if(nchim.gt.pnchim) then
		print*, 'nchim>pnchim =>stop'
		stop
	end if

!90	format(i3,14(1x,a4))
91	format(i3,/,15('	',a4))
92	format(i3,'	logt',100('	',i1,a6))

!**************************************** 
! INPUTS: nom_elem_bis, nchim

	call lisotop(nom_elem_bis,Mnucleo,Mzi,nchim)

! OUPUTS:
!   Mnucleo= atomic mass (12.0 for C12)
!   Mzi= atomic number (6 for C12, C13, C14)
!****************************************

	do i=1,nchim
		nucleo(i)=Mnucleo(i)
		zi(i)=Mzi(i)
	end do

	write(10,*) 'Atomic masses:'
	write(10,2001)(nucleo(i),i=1,nchim)

	write(10,*) 'Nucleus charges:'
	write(10,2001)(zi(i),i=1,nchim)

	mtot=Mmtot	! total mass of the star (g)
	rtot=Mrtot	! radius of the star(cm)
	ltot=Mltot	! luminosity of the star (Erg)
	teff=Mteff	! effective temperature (K)
	
	write(10,*) 
	write(10,*) 'mtot,rtot,ltot,teff:'
	write(10,2000) mtot,rtot,ltot,teff


      mtot = mtot * (1.98919000000001d0/1.9891d0) ! avoids a warning

	write(10,*) 

      CALL ctes_94 ()

	nomch='./'


!====================================================
! Initialization of the g_rad calculation. 
! This call has to be done once, at the first step of diffusion calculation. 
! This may correspond to the arrival on the ZAMS.
! The only data used here are the total mass (mtot) of the star and the list of elements.
! They allow to find the right tables of SVP parameters for the interpolations that are 
! necessary to determine the set of SVP parameters well fitted  to the mass of the star
! under consideration (mtot) and for elements considered in the model. In this 
! demo program, interpolations is only done when CHECK_INTER=2 (the code requests 
! this to the user). If  CHECK_INTER=1 or 3, the masses of models are the same as those of 
! the provided SVP tables, and so, there is no need to interpolate. 
! All other data provided by litcesam() or litother() called 
! above, can be updated at each time step of the evolution code, and be used for the 
! next call.

	version_svp='datai_SVP_v2'
      call initialize_SVP(version_svp)

!====================================================


! khi at surface
      write(10,'(//)')
      write(10,*) 'C_ref is the concentration of the element (vs H in number),'
      write(10,*) 'that have been used to established the SVP parameters.'
	write(10,*) 'Abundances: j,nom_elem,Mxchim,nucleo,ychim,C_ref'
	do j=1,nchim
		ychim(j)=Mxchim(j,1)/nucleo(j)
		if(id_ppxi(j).eq.1) then
			write(10,'(i5,3x,a4,4es12.5)')j,ADJUSTL(nom_elem(j)),
     +		Mxchim(j,1),nucleo(j),ychim(j),C_ref(j)
		else
			write(10,'(i5,3x,a4,3es12.5," NO SVP")')j,
     +		ADJUSTL(nom_elem(j)),Mxchim(j,1),nucleo(j),ychim(j)
		endif
	end do
      write(10,'(/)')
	write(10,'("List of khi!")')
      write(10,*) 'khi is the ratio: [concentration in this star]/C_ref'
      write(10,*) 'flag 0: there is no SVP parameter for that element.'
      write(10,*) '        khi is not used (set to 0.0) for that element.'
      write(10,'(/)')
	khi = 0.
	do j=1,nchim
		if(id_ppxi(j).eq.1) then
			do iso=1,nchim
				khi(j)=khi(j)+ychim(iso)/ychim(1)*isotops(j,iso)
			end do
			khi(j)=khi(j)/C_ref(j)
		else
			khi(j)=0.
		endif
		write(10,'(i2," khi_",a4," = ",es10.3,5x,i2)') j,
     +			ADJUSTL(nom_elem(j)),khi(j),id_ppxi(j)
	end do
	write(10,*)
	write(10,*) 'Accelerations (*g...), derivatives (*d...) ',
     +	'and ychim (*c...) [mass-fraction]:'

	write(10,92) nchim,
     + (id_ppxi(i),'*g'//ADJUSTL(nom_elem(i)),i=1,nchim),
     + (id_ppxi(i),'*d'//ADJUSTL(nom_elem(i)),i=1,nchim),
     + (id_ppxi(i),'*c'//ADJUSTL(nom_elem(i)),i=1,nchim)
	
! output for test
	write(20,93) ('lgr'//ADJUSTL(nom_elem(i)),
     +                'd'//ADJUSTL(nom_elem(i)),
     +                'chi'//ADJUSTL(nom_elem(i)),i=1,nchim)
93	format('logt',100('	',a7))

	print*,'itot=',itot

! start loop layers. Here, there no time step. 

	do i=1,itot
		mass = Mmass(i)	! mass from radius ray to center (g)
		ray  = Mray(i)	! distance of the layer from the star center (cm)
		t    = Mt(i)	! temperature (K)
		ro   = Mro(i)	! mass density (g cm-3)
		kap  = Mkap(i)	! Rosseland opacity
		grad = Mgrad(i)	! temperature radiative gradient
		lum  = Mlum(i)	! star luminosity (erg)
		nel  = Mnel(i)	! electrons density in number
		temp(i)=log10(t)
		do j=1,nchim
			ychim(j) = Mxchim(j,i)/nucleo(j)
			Chi(j,i) = (ychim(j)/ychim(1))/C_ref(j)
		end do


! Defined in modules: nchim, nom_elem, nucleo, mtot, rtot
!====================================================

! computation of g_rad (after the 2nd call, g_rad_SVP computes g_rad)
		Call g_rad_SVP(ray,t,nel,grav,ychim,g_rad,dg_rad)

!====================================================

	  	 write(10,2002) i,temp(i),
     +		(g_rad(j),j=1,nchim),(dg_rad(j,j),j=1,nchim),
     +		(ychim(j),j=1,nchim)

	   	write(20,'(f8.4,1p,100("	",d12.5))') temp(i),
     +		(lolog(g_rad(j)),dg_rad(j,j),Chi(j,i),j=1,nchim)

	end do

! end loop layers


	print*,'number of layers: ',itot
	write(10,*)'number of layers: ',itot

      Deallocate( nom_elem )
      
      print '(//)'
      print*, 'Run details are in svp_standalone.job'
      print*, 'Information about reading the model can be found in',
     +	' lect_osc.job or tgec_osc.job'
      print*, 'Run output details for svp_codes are in svp_initialize.job'
      print*
      print *,'Accelerations are computed and are in the file ctrlg_rad.dat !'
      print '(//)'

	STOP " End standalone "

	end

!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
	function lolog(A)

	real(KIND=KIND(0D0))   :: lolog,A

	if (A >0.) then
		lolog = log10(A)
	else
		lolog = -30.
	end if

	return
	end
!cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
	subroutine litcesam(
     &	nchim,nom_elem,
     &	mtot,rtot,ltot,teff,
     &	mass,ray,t,ro,kap,grad,lum,nel,
     &	xchim,np,itot)

! Reading of the model assigned to unit 1 (in main)

	implicit none

	integer pzi,pnchim,lglob,lm,lvar
	parameter (pzi=26,pnchim=50,lglob=50,lm=5000,lvar=100)

	integer	nchim,itot,j,k
! new .osc
	Integer, Parameter :: DP = KIND(1.d0)
	Integer :: i, nglob, nvar, nelem, niw, ios
	Real(DP), Dimension(lglob) :: glob
	Real(DP), Dimension(lvar,lm) :: var
	Real(DP), Dimension(lm) :: mue
	Real(DP)                :: somme,ceuler

	real*8	me,amu
	real*8	mtot,rtot,ltot,teff,
	1	mass(lm),ray(lm),t(lm),ro(lm),kap(lm),grad(lm),lum(lm),nel(lm),
	2	xchim(pnchim,lm),np(lm),xp(lm)
	
	character*1	skip
	character*4	nom_elem(pnchim)

!==============================================================================
	ceuler=2.718281828459045

	OPEN (UNIT = 11,
     &        FILE = 'lect_osc.job',
     &        STATUS = 'replace')


2000	format(1x,1p8d10.3)
2001	format(1x,1p15d10.3)
2002	format(1x,i4,1p,20('	',d11.4))
2004	format(1x,1p8d15.8)
	amu=1.6605402d-24	!masse atom. unite, Avogadro=1/amu
	me=9.1093897d-28	!masse electron


	read(1,'(a1)') skip
	read(1,'(a1)') skip
	read(1,'(a1)') skip
	read(1,'(a1)') skip
	read(1,90) nchim,(nom_elem(i),i=1,nchim)
	write(11,*) 'list considered elements:'
	write(11,91) nchim,(nom_elem(i),i=1,nchim)
*
	if(nchim.gt.pnchim) then
		print*, '(nchim.gt.pnchim) => STOP'
		stop
	end if

	Read(1,"(5I10)",IOStat=ios) itot, nglob, nvar, nelem, niw
	itot=itot-1 ! something works bad for the last layer. It is removed.
	print*,'itot, nglob, nvar, nelem, niw: ',itot, nglob, nvar, nelem, niw

!	glob: glogbal variables in mon_modele-ad.osc
!		glob(1)=mstar*msol
!		glob(2)=rtot*rsol
!		glob(3)=ltot*lsol
!		glob(4)=z0
!		glob(5)=x0
!		glob(6)=alpha
!		glob(7)=X dans ZC
!		glob(8)=Y dans ZC
!		glob(9)=d2p
!		glob(10)=d2ro
!		glob(11)=age
!		glob(12)=wrot vitesse de rotation globale
!		glob(13)=w_rot initial
!
!	var: variables per layer
!
!		var(1,i)=r*rsol
!	 	var(2,i)=log(m/mstar) -1.d38 au centre (en népérien?)
!		var(3,i)=t
!		var(4,i)=Ptot
!		var(5,i)=ro
!		var(6,i)=gradient reel d ln T / d ln P
!		var(7,i)=l
!		var(8,i)=kap
!		var(9,i)=energie thermo+gravifique
!		var(10,i)= Gamma1
!		var(11,i)= adiabatique gradient
!		var(12,i)=delta
!		var(13,i)=cp
!		var(14,i)=mu elec.
!		var(15,i)=vaissala, 0 at centre
!	 	var(16,i)=angular velocity, radian/sec
!	 	var(17,i)=d ln kappa / d ln T
!	 	var(18,i)=d ln kappa / d ln ro
!	 	var(19,i)=d epsilon(nuc) / d ln T
!	 	var(20,i)=d epsilon(nuc) / d ln ro
!		var(21,i)=Ptot / Pgas
!		var(22,i)= radiative gradient
!		...
!	  	var(nvar+j,i)=xchim(j), j=1,nchim 
! Ti47 is not titanium but all the remaining metals.
!
!---------------------------------------------------------------
!
	Read(1,"(5ES19.12)",IOStat=ios) (glob(i),i=1,nglob)
	
	mtot = glob(1)
	rtot = glob(2)
	ltot = glob(3)
	teff = (ltot/(4.*acos(-1.d0)*(rtot**2)*5.6692e-5))**(0.25)
	write(11,*) 'mtot,rtot,ltot,teff:'
	write(11,2000)mtot,rtot,ltot,teff
	
!	print*,'loop 1,itot=',itot
	Do i = 1 , itot
	   Read(1,"(5ES19.12)",IOStat=ios) (var(j,i),j=1,nvar+nchim)
	   If ( ios /= 0 ) STOP " Pb-09 "
	End Do
	print*

	write(11,'("i,mass,ray,t,ro,kap,grad,lum,nel")')
	Do i = 1 , itot   
	   ray(i)   = var(1,i)                !  cm
	   mass(i)  = 1. - ceuler**var(2,i)   ! (1-m/mtot) 
	   t(i)     = var(3,i)                !  K
	   ro(i)    = var(5,i)                !  g cm-3
	   lum(i)   = var(7,i)                !  erg
	   mue(i)   = var(14,i)               !  g mol-1  
	   nel(i)   = 6.0232e+23*ro(i)/mue(i) ! n_e = avogadro*dens / mu_e (cm-3)
	   grad(i)  = var(22,i)
	   kap(i)   = var(8,i)
	   write(11,2002)
     &	i,mass(i),ray(i),t(i),ro(i),kap(i),grad(i),lum(i),nel(i)
	End Do 
	
	write(11,'(//,"xchim:")')
	write(11,'(20("	",a4))')(nom_elem(i),i=1,nchim)
	Do i = 1 , itot
	   xchim(1:nchim,i) = var(nvar+1:nvar+nchim,i)  ! abundance: mass fraction of the element
	   somme = SUM(xchim(1:nchim,i))
	   If ( ABS( SUM(xchim(1:nchim,i)) - 1.0_DP ) >= 1.0e-13_DP ) 
     +	 Write(*,*) 'Sum xchim>1:',i,somme
	   write(11,2002) i, (xchim(j,i),j=1,nchim),somme
	   np(i)=xchim(1,i)*(ro(i)-nel(i)*me)/amu   ! a ce stade: np=nH
	End Do

	
	Do i = 1 , itot
		call  calcNP(t(i),nel(i),xp(i))
		np(i)=np(i)*xp(i)
	End Do


90	format(i3,14(1x,a4),/,14(1x,a4))
91	format(i3,14('	',a4),/,14('	',a4))


10	close(unit=1)
c	itot=i-1
	print*,'Layers number itot: ',itot
	close(unit=11)
	return

	end
!=======================================================================
	subroutine lisotop(nom_elem_bis,Mnucleo,Mzi,nchim)

! Reads isotopes list (isotopes.data) to connect with nom_elem

	Implicit None
	
	integer, parameter                      :: pzi=26,pnchim=50

	integer                                 :: nchim
	real(KIND=KIND(0D0)), dimension(pnchim) :: Mnucleo,Mzi
	Character(Len=4), dimension(pnchim)     :: nom_elem_bis

! Variables locales
	real(KIND=KIND(0D0)), dimension(200)  :: nucleoL,ziL
	Character(Len=4), dimension(200)      :: nom_elem_L
	
	integer                               :: i,is,j
	
	OPEN (UNIT = 2,
     +      FILE = 'data_standalone/isotopes.dat',
     +      STATUS = 'old')
	is = 0
	do i=1,200
		read(2,'(a4,d18.11,d5.0)',end=10) nom_elem_L(i),nucleoL(i),ziL(i)
		is=is+1
	end do
10	continue

	do j=1,nchim
		do i=1,is
			if(ADJUSTL(nom_elem_bis(j)).eq.nom_elem_L(i)) then
				Mnucleo(j)=nucleoL(i)
				Mzi(j)=ziL(i)
				cycle
			end if
		end do
	end do

	close(2)

	return
	end
!=======================================================================
	subroutine calcNP(temp,nel,xp)

! Computes H ionisation (taking density effect into acompte).
! From CESAM code,

	Implicit None
	Integer, Parameter :: DP = KIND(1.d0) , ndim = 10
!
	Real(DP), Parameter :: lambda_compton = 2426.310215e-13_DP    !  cm
	Real(DP), Parameter :: mec2 = 510998.9_DP                     !  eV
	Real(DP), Parameter :: k_eV = 8.617342e-05_DP                 ! n eV/K
!
!
	Real(DP), Parameter :: ioni_hydrogene = 13.595_DP 
	Real(DP), Parameter :: Rydberg        = 13.60569172_DP        ! eV
!
	Real(DP), Parameter :: Pi = 3.141592653589793238_DP
	Real(DP), Parameter :: avogadro = 6.022e+23_dp
!
	Real(DP) :: conversion, lambda_therm, expo, G = 1.0_DP , K    ! G ratio of statistical weights
!
! Corrections for EFF ionization : see Eggleton et al. A&A 23 (1973) 325
!
	Real(DP), Parameter :: z_barre_ini = 1.0_DP 
	Real(DP) :: z_barre, a0_cube, xhi0, delta_mu

! IN/OUT
	Real(DP) :: xp, temp, nel

	conversion = lambda_compton / SQRT( 2.0_DP * Pi * k_ev / mec2 )

	lambda_therm = conversion / SQRT( temp )
	K = lambda_therm**3 * nel / G
	expo = EXP( + ioni_hydrogene / ( k_ev * temp ) )
!
! estimation of initial correction to have z_barre value
!
	z_barre = z_barre_ini                  ! mean nucleus charge 
	a0_cube = (0.523e-8_DP/z_barre)**3     !  cm-3
	xhi0 = 13.5_DP * z_barre**2            !  eV
	delta_mu = a0_cube * nel * ( 1.0_DP + (20.0_DP*xhi0/k_ev) / temp )     ! correction EFF
	xp = 1.0_DP / ( 1.0_DP + K * expo * EXP( - delta_mu) )                 ! fraction of n(H)
!
! estimation of correction to have new z_barre value
!
	z_barre = 1.0_DP - xp
	a0_cube = (0.523e-8_DP/z_barre)**3     ! en cm-3
	xhi0 = 13.5_DP * z_barre**2            ! en eV
	delta_mu = a0_cube * nel * ( 1.0_DP + (20.0_DP*xhi0/k_ev) / temp )     ! correction EFF
	if(delta_mu.lt.300.) then   ! fraction n(H)
		xp = 1.0_DP / ( 1.0_DP + K * expo * EXP( - delta_mu) )  
!		print'(10es8.1)',K,expo, EXP( - delta_mu)
	else
		xp = 1.0_DP 
	endif
		               
	
!
! 
	return
	end
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
	subroutine littgec2(
     &	nchim,nom_elem,
     &	mtot,rtot,ltot,teff,
     &	mass,ray,t,ro,kap,grad,lum,nel,
     &	xchim,np,itot,nbncl)

! Reading of the model assigned to unit 1 and 2 (in main)
! G.Alecian Janvier 2026

	implicit none

	integer pzi,pnchim,lglob,lm,lvar
	parameter (pzi=26,pnchim=50,lglob=50,lm=5000,lvar=100)

	integer	nchim,itot,j,k
! new .osc
	Integer, Parameter :: DP = KIND(1.d0)
	Integer :: i, nglob, nvar, nelem, niw, ios,nbncl
	Real(DP), Dimension(lglob) :: glob
	Real(DP), Dimension(lvar,lm) :: var
	Real(DP), Dimension(lm) :: mue
	Real(DP)                :: somme,ceuler

	real*8	me,amu
	real*8	mtot,rtot,ltot,teff,
     +	mass(lm),ray(lm),t(lm),ro(lm),kap(lm),grad(lm),lum(lm),nel(lm),
     +	xchim(pnchim,lm),np(lm),xp(lm)
	
	character*1	skip
	character*4	nom_elem(pnchim)

	character*15 nom_col(100),A
	character*1000 ligne_titre

!==============================================================================
	ceuler=2.718281828459045

2000	format(1x,1p8d10.3)
2001	format(1x,1p15d10.3)
2002	format(1x,i4,1p,50('	',d11.4))
2004	format(1x,1p8d15.8)

	OPEN (UNIT = 11,
     &        FILE = 'lect_tgec.job',
     &        STATUS = 'replace')

	read(1,*)
	do i=2,22
		read(1,"(18x,ES15.8)") glob(i)
		print'(i3,1pD15.8)',i,glob(i)
	end do

!	Read(1,"(5ES19.12)",IOStat=ios) (glob(i),i=1,nglob)
!	
! xtg
! 1	Model number     
! 2	Mass [g]         
! 3	Solar masses [Mo]
! 4	X (initial)      
! 5	Z (initial)      
! 6	AlphaOvershooting
! 7	Age [yr]         
! 8	Log(Te)          
! 9	Log(L/Lo)        
! 10	Lg/L             
! 11	Radius [cm]      
! 12	Log(g)           
! 13	Tc [K]           
! 14	rhoc [g/cm3]     
! 15	rhomoyen [g/cm3] 
! 16	rhoc/<rho>       
! 17	Xc               
! 18	Mnc [g] no over  
! 19	Mnco [g] over    
! 20	Minertia [g.cm2] 
! 21	Integral |n|/r.dr
! 22	tdyn [s]   
     
	mtot = glob(2)
	rtot = glob(11)
	ltot = (10**glob(9))*3.846d33
	teff = (ltot/(4.*acos(-1.d0)*(rtot**2)*5.6692e-5))**(0.25)
	write(11,*) 'mtot,rtot,ltot,teff:'
	write(11,2000)mtot,rtot,ltot,teff

!Pour TGEC xtl, en supposant que nbncl=44
!---------------------
! 1:j
! 2:r
! 3:M
! 4:rho
! 5:L
! 6:T
! 7:g
! 8:kappa
! 9:Ne
! 10:P
! 11:X
! 12:Y
! 13:r/R
! 14:logdm/M
! 15:nH
! 16:n He3
! 17:n He4
! 18:n Li6
! 19:n Li7
! 20:n Be9
! 21:n B10
! 22:n B11
! 23:n C12
! 24:n C13
! 25:n N14
! 26:n N15
! 27:n O16
! 28:n O17
! 29:n O18
! 30:nNe20
! 31:nNe22
! 32:nMg24
! 33:nMg25
! 34:nMg26
! 35:nCa40
! 36:nFe56
! 37:nNa23
! 38:nAl27
! 39:nSi28
! 40:n S32
! 41:nAr36
! 42:nHI
! 43:nHII
! 44:nnucl
	itot=0
	Do i = 1,5000
	   Read(2,*,IOStat=ios,end=100)
	   itot=itot+1
	   If ( ios /= 0 ) STOP " Pb-09 tgec2"
	End Do
100	rewind (2)
	itot=itot-1
	print*, 'itot=',itot
	
	nvar=14
	nchim=nbncl-3-nvar
	
	print*,'nchim=',nchim

	if(nchim.gt.pnchim) then
		print*, 'nchim.gt.pnchim in subroutine littgec2 => STOP'
		stop
	end if
	
!----------------------------------------- recuperation nom colonnes
	read(2,'(a)') ligne_titre
	ligne_titre=TRIM(ADJUSTL(ligne_titre))
	k=1
	nom_col=""
	do i=1,1000
		if(ichar(ligne_titre(i:i)).eq.9) then ! si TAB
			k=k+1
		else
			nom_col(k)=TRIM(ADJUSTL(nom_col(k)))//ligne_titre(i:i)
		end if
		if(k.gt.nbncl) exit
	end do
	
	do j=1,nbncl
		A=TRIM(nom_col(j))
		if(j.le.14) then
			nom_col(j)=TRIM(A(INDEX(A,":")+1:LEN_TRIM(A)))
		else
			nom_col(j)=TRIM(A(INDEX(A,":")+2:LEN_TRIM(A)))
			if(j.le.14+nchim) then 
				nom_elem(j-14)=nom_col(j)
			endif
		end if
		print*,nom_col(j)
	end do
	nom_elem(1)=TRIM(nom_elem(1))//'1'   !correction masse atomique manquante dans .xtg
	print*,'noms elem: ',(nom_elem(j),', ',j=1,nchim)
!-----------------------------------------


	
	Do i = 1 ,itot
	   Read(2,*,IOStat=ios,end=101) k,(var(j,i),j=2,nvar+nchim+3) 
	   If ( ios /= 0 ) then
	   	print*, 'ios,i,k,(var(j,i),j=2,nvar+nchim+3):   ',
     +		ios,i,k,(var(j,i),j=2,nvar+nchim+3)
	   	STOP " Pb-10 tgec2"
	   end if
	End Do
101	continue
	print*

	write(11,'("i,mass,ray,t,ro,kap,grad,lum,nel")')
	Do i = 1 , itot   
	   ray(i)   = var(2,i)                !  cm
!	   mass(i)  = 1. - ceuler**var(2,i)   ! (1-m/mtot) 
	   mass(i)  = 10**var(14,i)   	! (1-m/mtot) 
	   t(i)     = var(6,i)                !  K
	   ro(i)    = var(4,i)                !  g cm-3
	   lum(i)   = var(5,i)                !  erg
!	   mue(i)   = var(14,i)               !  g mol-1  
	   mue(i)   = 0.               !  g mol-1  
!	   nel(i)   = 6.0232e+23*ro(i)/mue(i) ! n_e = avogadro*dens / mu_e (cm-3)
	   nel(i)   = var(9,i)                ! n_e
	   grad(i)  = 0.
	   kap(i)   = var(8,i)
	   np(i)	= var(nbncl-1,i)
	   write(11,2002)
     &	i,mass(i),ray(i),t(i),ro(i),kap(i),grad(i),lum(i),nel(i),np(i)
	End Do 
	
	write(11,'(//,"xchim:")')
	write(11,'(50("	",a4))')(nom_elem(i),i=1,nchim),'summ'
	Do i = 1 , itot
	   xchim(1:nchim,i) = var(nvar+1:nvar+nchim,i)  ! abundance: mass fraction of the element
	   somme = SUM(xchim(1:nchim,i))
!	   If ( ABS( SUM(xchim(1:nchim,i)) - 1.0_DP ) >= 1.0e-13_DP ) 
!     +	 Write(*,*) 'Sum xchim>1:',i,somme
	   write(11,2002) i, (xchim(j,i),j=1,nchim),somme
!	   np(i)=xchim(1,i)*(ro(i)-nel(i)*me)/amu   ! a ce stade: np=nH
	End Do


	print*, 'sortie littgec2'
	
	return
	end


